chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === "solve_securely") {
        handleSecureSolve(request)
            .then(result => sendResponse(result))
            .catch(error => sendResponse({ success: false, error: error.message }));
        return true; 
    }
});

async function handleSecureSolve(data) {
    const { payload, token, model, serverUrl } = data;
    
    // 1. Get Key from Server (Mocking or using your existing PHP logic)
    // For this DEMO, we assume the serverUrl logic is identical to your Pearson one.
    
    let authResponse;
    try {
        authResponse = await fetch(`${serverUrl}/solve.php`, {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${token}`,
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({}) 
        });
    } catch (e) {
        throw new Error("Failed to contact server.");
    }

    if (!authResponse.ok) throw new Error("Server Error: " + authResponse.status);
    const authData = await authResponse.json();
    
    if (!authData.success || !authData.api_key) throw new Error(authData.error || "No API Key returned");

    // 2. Call Gemini
    const answers = await solveWithGemini(payload, authData.api_key, model);
    
    return { 
        success: true, 
        answers: answers, 
        creditsLeft: authData.requests_left 
    };
}

async function solveWithGemini(payload, apiKey, modelName) {
    const prompt = `
    You are an expert Cambridge English tutor.
    TASK: Fill in the blanks based on the context.
    
    CONTEXT:
    """
    ${payload.text}
    """

    BLANKS TO SOLVE:
    ${JSON.stringify(payload.blanks)}

    INSTRUCTIONS:
    1. Return valid JSON only: {"ID": "ANSWER", "ID2": "ANSWER"}
    2. If options are provided, choose the exact string from the options.
    `;

    const url = `https://generativelanguage.googleapis.com/v1beta/models/${modelName || 'gemini-2.5-flash'}:generateContent?key=${apiKey}`;
    
    const response = await fetch(url, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
            contents: [{ parts: [{ text: prompt }] }]
        })
    });

    const data = await response.json();
    
    if (!data.candidates || !data.candidates[0].content) throw new Error("Gemini returned empty.");
    
    let rawText = data.candidates[0].content.parts[0].text;
    rawText = rawText.replace(/```json/g, '').replace(/```/g, '').trim();
    
    try {
        return JSON.parse(rawText);
    } catch (e) {
        // Simple heuristic fix for JSON parsing if Gemini chats a bit
        const start = rawText.indexOf('{');
        const end = rawText.lastIndexOf('}');
        if (start !== -1 && end !== -1) {
            return JSON.parse(rawText.substring(start, end + 1));
        }
        throw new Error("JSON Parse failed");
    }
}